import { Injectable, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../prisma/prisma.service';

@Injectable()
export class AdminService {
  constructor(private prisma: PrismaService) {}

  async listTenants(page = 1, limit = 20) {
    const skip = (page - 1) * limit;
    const [data, total] = await Promise.all([
      this.prisma.tenant.findMany({
        where: { deletedAt: null },
        skip,
        take: limit,
        orderBy: { createdAt: 'desc' },
        include: {
          subscription: true,
          _count: { select: { products: true, locations: true } },
        },
      }),
      this.prisma.tenant.count({ where: { deletedAt: null } }),
    ]);
    return {
      data,
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit),
    };
  }

  async getTenant(tenantId: string) {
    const t = await this.prisma.tenant.findFirst({
      where: { id: tenantId, deletedAt: null },
      include: {
        subscription: true,
        users: { include: { user: { select: { id: true, email: true, name: true } } } },
        _count: { select: { products: true, locations: true } },
      },
    });
    if (!t) throw new NotFoundException('Tenant bulunamadı');
    return t;
  }

  async softDeleteTenant(tenantId: string) {
    await this.prisma.tenant.update({
      where: { id: tenantId },
      data: { deletedAt: new Date() },
    });
    return { ok: true };
  }

  async getBasicAnalytics() {
    const [tenantCount, productCount, activeSubs] = await Promise.all([
      this.prisma.tenant.count({ where: { deletedAt: null } }),
      this.prisma.product.count(),
      this.prisma.subscription.count({ where: { status: { in: ['active', 'trialing'] } } }),
    ]);
    return { tenantCount, productCount, activeSubscriptions: activeSubs };
  }
}
