import { Injectable, UnauthorizedException, ConflictException } from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import * as bcrypt from 'bcrypt';
import { v4 as uuidv4 } from 'uuid';
import * as crypto from 'crypto';
import { PrismaService } from '../prisma/prisma.service';
import { slugify } from '@3dqr/shared';
import type { RegisterBody, LoginBody } from '@3dqr/shared';
import type { UserRole } from '@prisma/client';

const REFRESH_EXPIRY_DAYS = 7;
const REFRESH_COOKIE_NAME = 'refreshToken';

@Injectable()
export class AuthService {
  constructor(
    private prisma: PrismaService,
    private jwt: JwtService,
  ) {}

  async register(body: RegisterBody) {
    const existing = await this.prisma.user.findUnique({ where: { email: body.email } });
    if (existing) throw new ConflictException('Bu e-posta zaten kayıtlı');

    const slug = body.tenantSlug || slugify(body.tenantName);
    const tenantExists = await this.prisma.tenant.findUnique({ where: { slug } });
    if (tenantExists) throw new ConflictException('Bu restoran slug’ı kullanılıyor');

    const passwordHash = await bcrypt.hash(body.password, 10);
    const user = await this.prisma.$transaction(async (tx) => {
      const u = await tx.user.create({
        data: {
          email: body.email,
          passwordHash,
          name: body.name,
        },
      });
      const tenant = await tx.tenant.create({
        data: { name: body.tenantName, slug },
      });
      await tx.userTenant.create({
        data: { userId: u.id, tenantId: tenant.id, role: 'owner' },
      });
      await tx.subscription.create({
        data: { tenantId: tenant.id, status: 'canceled' },
      });
      return { ...u, tenantId: tenant.id };
    });

    const tokens = await this.issueTokens(user.id, user.email, 'owner', user.tenantId);
    return { user: this.sanitizeUser(user), ...tokens };
  }

  async login(body: LoginBody) {
    const user = await this.prisma.user.findUnique({
      where: { email: body.email },
      include: {
        tenants: { include: { tenant: true }, take: 1 },
      },
    });
    if (!user || !(await bcrypt.compare(body.password, user.passwordHash))) {
      throw new UnauthorizedException('E-posta veya şifre hatalı');
    }
    const firstTenant = user.tenants[0];
    const tenantId = firstTenant?.tenantId;
    const role = (firstTenant?.role ?? 'editor') as UserRole;
    const tokens = await this.issueTokens(user.id, user.email, role, tenantId);
    return {
      user: this.sanitizeUser({ ...user, tenantId }),
      ...tokens,
    };
  }

  async refresh(refreshToken: string) {
    if (!refreshToken) throw new UnauthorizedException('Refresh token gerekli');
    const hash = crypto.createHash('sha256').update(refreshToken).digest('hex');
    const tokenRow = await this.prisma.refreshToken.findFirst({
      where: { tokenHash: hash, expiresAt: { gt: new Date() } },
      include: { user: { include: { tenants: { include: { tenant: true }, take: 1 } } } },
    });
    if (!tokenRow) throw new UnauthorizedException('Geçersiz veya süresi dolmuş token');
    await this.prisma.refreshToken.delete({ where: { id: tokenRow.id } });
    const ut = tokenRow.user.tenants[0];
    const tokens = await this.issueTokens(
      tokenRow.user.id,
      tokenRow.user.email,
      (ut?.role ?? 'editor') as UserRole,
      ut?.tenantId,
    );
    return {
      user: this.sanitizeUser({ ...tokenRow.user, tenantId: ut?.tenantId }),
      ...tokens,
    };
  }

  async logout(refreshToken: string) {
    if (!refreshToken) return;
    const hash = crypto.createHash('sha256').update(refreshToken).digest('hex');
    await this.prisma.refreshToken.deleteMany({ where: { tokenHash: hash } });
  }

  private async issueTokens(
    userId: string,
    email: string,
    role: UserRole,
    tenantId?: string,
  ) {
    const payload = { sub: userId, email, role, tenantId };
    const accessToken = this.jwt.sign(payload, { expiresIn: '15m' });
    const refresh = uuidv4() + crypto.randomBytes(24).toString('hex');
    const refreshHash = crypto.createHash('sha256').update(refresh).digest('hex');
    const expiresAt = new Date();
    expiresAt.setDate(expiresAt.getDate() + REFRESH_EXPIRY_DAYS);
    await this.prisma.refreshToken.create({
      data: { userId, tokenHash: refreshHash, expiresAt },
    });
    return {
      accessToken,
      refreshToken: refresh,
      expiresIn: 900,
    };
  }

  getRefreshCookieName() {
    return REFRESH_COOKIE_NAME;
  }

  getRefreshCookieOptions() {
    return {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax' as const,
      maxAge: REFRESH_EXPIRY_DAYS * 24 * 60 * 60 * 1000,
      path: '/',
    };
  }

  private sanitizeUser(u: { id: string; email: string; name: string; tenantId?: string; platformRole?: unknown }) {
    return {
      id: u.id,
      email: u.email,
      name: u.name,
      tenantId: u.tenantId,
      platformRole: u.platformRole,
    };
  }
}
