import { Controller, Get, Param, Query, Req, Res, UseGuards } from '@nestjs/common';
import { Response } from 'express';
import { JwtAuthGuard, TenantGuard, SubscriptionGuard } from '../auth/guards';
import { TenantService } from './tenant.service';
import { QrService } from './qr.service';

@Controller('tenant')
@UseGuards(JwtAuthGuard, TenantGuard, SubscriptionGuard)
export class QrController {
  constructor(
    private tenant: TenantService,
    private qr: QrService,
  ) {}

  @Get('qr/table/:tableId')
  async getTableQr(
    @Param('tableId') tableId: string,
    @Query('format') format: 'png' | 'svg' = 'png',
    @Req() req: any,
    @Res() res: Response,
  ) {
    const tenantId = req.tenantId;
    const tables = await this.tenant.listTables(tenantId);
    const table = tables.find((t) => t.id === tableId);
    if (!table) return res.status(404).send('Masa bulunamadı');
    const menuBaseUrl = process.env.MENU_PUBLIC_URL || 'https://menu.domain.com';
    const url = `${menuBaseUrl}/r/${(await this.tenant.getTenantForUser(req.user.id, tenantId)).slug}/l/${table.locationId}/t/${tableId}`;
    const buffer = format === 'svg' ? await this.qr.toSvg(url) : await this.qr.toPng(url);
    res.setHeader('Content-Type', format === 'svg' ? 'image/svg+xml' : 'image/png');
    res.setHeader('Content-Disposition', `inline; filename="qr-${table.name}.${format}"`);
    res.send(buffer);
  }

  @Get('qr/location/:locationId')
  async getLocationQr(
    @Param('locationId') locationId: string,
    @Query('format') format: 'png' | 'svg' = 'png',
    @Req() req: any,
    @Res() res: Response,
  ) {
    const tenantId = req.tenantId;
    const locations = await this.tenant.listLocations(tenantId);
    const loc = locations.find((l) => l.id === locationId);
    if (!loc) return res.status(404).send('Şube bulunamadı');
    const tenant = await this.tenant.getTenantForUser(req.user.id, tenantId);
    const menuBaseUrl = process.env.MENU_PUBLIC_URL || 'https://menu.domain.com';
    const url = `${menuBaseUrl}/r/${tenant.slug}/l/${locationId}`;
    const buffer = format === 'svg' ? await this.qr.toSvg(url) : await this.qr.toPng(url);
    res.setHeader('Content-Type', format === 'svg' ? 'image/svg+xml' : 'image/png');
    res.setHeader('Content-Disposition', `inline; filename="qr-${loc.name}.${format}"`);
    res.send(buffer);
  }

  @Get('qr/tenant')
  async getTenantQr(
    @Query('format') format: 'png' | 'svg' = 'png',
    @Req() req: any,
    @Res() res: Response,
  ) {
    const tenantId = req.tenantId;
    const tenant = await this.tenant.getTenantForUser(req.user.id, tenantId);
    const menuBaseUrl = process.env.MENU_PUBLIC_URL || 'https://menu.domain.com';
    const url = `${menuBaseUrl}/r/${tenant.slug}`;
    const buffer = format === 'svg' ? await this.qr.toSvg(url) : await this.qr.toPng(url);
    res.setHeader('Content-Type', format === 'svg' ? 'image/svg+xml' : 'image/png');
    res.setHeader('Content-Disposition', `inline; filename="qr-${tenant.slug}.${format}"`);
    res.send(buffer);
  }
}
