import { Injectable, BadRequestException } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { PrismaService } from '../prisma/prisma.service';
import { getSignedUrl } from '@aws-sdk/s3-request-presigner';
import { S3Client, PutObjectCommand } from '@aws-sdk/client-s3';
import { getAllowedMimes, MAX_IMAGE_SIZE_BYTES, MAX_MODEL_SIZE_BYTES } from '@3dqr/shared';

@Injectable()
export class UploadService {
  private s3: S3Client;
  private bucket: string;
  private cdnUrl: string;

  constructor(
    private config: ConfigService,
    private prisma: PrismaService,
  ) {
    this.s3 = new S3Client({
      region: this.config.get<string>('AWS_REGION', 'eu-west-1'),
      credentials: this.config.get('AWS_ACCESS_KEY_ID')
        ? {
            accessKeyId: this.config.get<string>('AWS_ACCESS_KEY_ID')!,
            secretAccessKey: this.config.get<string>('AWS_SECRET_ACCESS_KEY')!,
          }
        : undefined,
    });
    this.bucket = this.config.get<string>('AWS_S3_BUCKET', '');
    this.cdnUrl = this.config.get<string>('CDN_URL', '').replace(/\/$/, '');
  }

  async getPresignedUrl(
    tenantId: string,
    userId: string,
    productId: string,
    filename: string,
    contentType: string,
    type: 'image' | 'model3d',
  ): Promise<{ uploadUrl: string; publicUrl: string }> {
    const product = await this.prisma.product.findFirst({
      where: { id: productId, tenantId },
    });
    if (!product) throw new BadRequestException('Ürün bulunamadı');

    const allowed = getAllowedMimes(type);
    if (!allowed.includes(contentType)) {
      throw new BadRequestException(`Geçersiz dosya tipi. İzin verilen: ${allowed.join(', ')}`);
    }

    const ext = type === 'image' ? filename.split('.').pop() ?? 'jpg' : 'glb';
    if (!this.bucket) {
      throw new BadRequestException('Dosya yükleme yapılandırılmamış (AWS_S3_BUCKET)');
    }
    const key = `tenant/${tenantId}/products/${productId}/${type}-${Date.now()}.${ext}`;

    const command = new PutObjectCommand({
      Bucket: this.bucket,
      Key: key,
      ContentType: contentType,
    });

    const uploadUrl = await getSignedUrl(this.s3, command, { expiresIn: 3600 });
    const publicUrl = this.cdnUrl ? `${this.cdnUrl}/${key}` : `https://${this.bucket}.s3.amazonaws.com/${key}`;

    return { uploadUrl, publicUrl };
  }
}
