'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import { usePathname, useRouter } from 'next/navigation';
import { tenantApi, authApi } from '@/lib/api';

const nav = [
  { href: '/dashboard', label: 'Özet' },
  { href: '/dashboard/products', label: 'Ürünler' },
  { href: '/dashboard/categories', label: 'Kategoriler' },
  { href: '/dashboard/locations', label: 'Şubeler & Masalar' },
  { href: '/dashboard/qr', label: 'QR Kodlar' },
];

export default function DashboardLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  const pathname = usePathname();
  const router = useRouter();
  const [tenant, setTenant] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    tenantApi
      .getMe()
      .then((data) => {
        if (!data?.id) {
          router.replace('/login');
          return;
        }
        setTenant(data);
      })
      .catch(() => router.replace('/login'))
      .finally(() => setLoading(false));
  }, [router]);

  const handleLogout = async () => {
    await authApi.logout();
    localStorage.removeItem('accessToken');
    router.replace('/login');
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-10 w-10 border-2 border-emerald-500 border-t-transparent" />
      </div>
    );
  }

  if (!tenant) return null;

  const subscriptionActive =
    tenant.subscription?.status === 'active' || tenant.subscription?.status === 'trialing';

  return (
    <div className="min-h-screen bg-slate-50 flex">
      <aside className="w-56 bg-white border-r border-slate-200 flex flex-col">
        <div className="p-4 border-b border-slate-200">
          <h2 className="font-semibold text-slate-800">3DQR</h2>
          <p className="text-sm text-slate-500 truncate">{tenant.name}</p>
        </div>
        <nav className="p-2 flex-1">
          {nav.map((item) => (
            <Link
              key={item.href}
              href={item.href}
              className={`block px-3 py-2 rounded-lg text-sm font-medium ${
                pathname === item.href
                  ? 'bg-emerald-50 text-emerald-700'
                  : 'text-slate-600 hover:bg-slate-100'
              }`}
            >
              {item.label}
            </Link>
          ))}
        </nav>
        <div className="p-2 border-t border-slate-200">
          <button
            onClick={handleLogout}
            className="w-full text-left px-3 py-2 rounded-lg text-sm text-slate-600 hover:bg-slate-100"
          >
            Çıkış
          </button>
        </div>
      </aside>
      <main className="flex-1 p-6 overflow-auto">
        {!subscriptionActive && (
          <div className="mb-4 p-4 bg-amber-50 border border-amber-200 rounded-lg text-amber-800 text-sm">
            Aboneliğiniz aktif değil. Ürün yönetimi için lütfen ödeme sayfasından aboneliği yenileyin.
          </div>
        )}
        {children}
      </main>
    </div>
  );
}
