const API_URL = process.env.NEXT_PUBLIC_API_URL || 'http://localhost:4000';

function getToken(): string | null {
  if (typeof window === 'undefined') return null;
  return localStorage.getItem('accessToken');
}

export async function api<T>(
  path: string,
  options: RequestInit = {},
): Promise<T> {
  const token = getToken();
  const res = await fetch(`${API_URL}${path}`, {
    ...options,
    headers: {
      'Content-Type': 'application/json',
      ...(token && { Authorization: `Bearer ${token}` }),
      ...options.headers,
    },
    credentials: 'include',
  });
  if (!res.ok) {
    const err = await res.json().catch(() => ({ message: res.statusText }));
    throw new Error(err.message || 'İstek başarısız');
  }
  return res.json();
}

export const authApi = {
  login: (email: string, password: string) =>
    api<{ user: any; accessToken: string; expiresIn: number }>('/auth/login', {
      method: 'POST',
      body: JSON.stringify({ email, password }),
    }),
  register: (body: {
    email: string;
    password: string;
    name: string;
    tenantName: string;
    tenantSlug: string;
  }) =>
    api<{ user: any; accessToken: string; expiresIn: number }>('/auth/register', {
      method: 'POST',
      body: JSON.stringify(body),
    }),
  refresh: () =>
    api<{ user: any; accessToken: string; expiresIn: number }>('/auth/refresh', {
      method: 'POST',
    }),
  logout: () => api<{ ok: boolean }>('/auth/logout', { method: 'POST', credentials: 'include' }),
};

export const tenantApi = {
  getMe: () => api<any>('/tenant/me'),
  listLocations: () => api<any[]>('/tenant/locations'),
  createLocation: (body: any) => api<any>('/tenant/locations', { method: 'POST', body: JSON.stringify(body) }),
  updateLocation: (id: string, body: any) => api<any>(`/tenant/locations/${id}`, { method: 'PUT', body: JSON.stringify(body) }),
  deleteLocation: (id: string) => api<void>(`/tenant/locations/${id}`, { method: 'DELETE' }),
  listTables: (locationId?: string) =>
    api<any[]>(`/tenant/tables${locationId ? `?locationId=${locationId}` : ''}`),
  createTable: (body: any) => api<any>('/tenant/tables', { method: 'POST', body: JSON.stringify(body) }),
  updateTable: (id: string, body: any) => api<any>(`/tenant/tables/${id}`, { method: 'PUT', body: JSON.stringify(body) }),
  deleteTable: (id: string) => api<void>(`/tenant/tables/${id}`, { method: 'DELETE' }),
};

export const productApi = {
  listCategories: () => api<any[]>('/products/categories'),
  createCategory: (body: any) => api<any>('/products/categories', { method: 'POST', body: JSON.stringify(body) }),
  updateCategory: (id: string, body: any) => api<any>(`/products/categories/${id}`, { method: 'PUT', body: JSON.stringify(body) }),
  deleteCategory: (id: string) => api<void>(`/products/categories/${id}`, { method: 'DELETE' }),
  listProducts: (params?: { categoryId?: string }) => {
    const q = new URLSearchParams(params as any).toString();
    return api<any[]>(`/products${q ? `?${q}` : ''}`);
  },
  getProduct: (id: string) => api<any>(`/products/${id}`),
  createProduct: (body: any) => api<any>('/products', { method: 'POST', body: JSON.stringify(body) }),
  updateProduct: (id: string, body: any) => api<any>(`/products/${id}`, { method: 'PUT', body: JSON.stringify(body) }),
  deleteProduct: (id: string) => api<void>(`/products/${id}`, { method: 'DELETE' }),
  bulkAction: (body: any) => api<{ updated: number }>('/products/bulk', { method: 'POST', body: JSON.stringify(body) }),
};

export const uploadApi = {
  getPresigned: (body: { productId: string; filename: string; contentType: string; type: 'image' | 'model3d' }) =>
    api<{ uploadUrl: string; publicUrl: string }>('/upload/presigned', { method: 'POST', body: JSON.stringify(body) }),
};
